async function loadState() {
  return await chrome.storage.local.get({
    enabled: false,
    selectedId: null,
    proxies: [],
    health: {},
    loggedIn: false,
    email : "",
    username: "",
    token: "",
    vip: "0",
    speed: "N/A",
    quota: "N/A"
  });
}


// chrome.runtime.onMessage.addListener((msg) => {
//   if (msg.type === "AUTH_UPDATED" || msg.type === "PROXY_LIST_UPDATED") {
//     // loadState().then(st => {
//     //   render(st);
//     // });
//     chrome.runtime.sendMessage({ type: "REFRESH" }).then(() => {
//         refreshUI();
//     });
//   }
// });
async function checkProxyConflict() {
  const noticeEl = document.getElementById("conflictNotice");
  if (!noticeEl) return;

  chrome.proxy.settings.get({ incognito: false }, (details) => {
    // levelOfControl 有四种状态：
    // 'controlled_by_other_extensions': 被其他插件控制
    // 'controllable_by_this_extension': 当前插件可以控制
    // 'controlled_by_this_extension': 当前插件正在控制
    
    if (details.levelOfControl === 'controlled_by_other_extensions') {
      noticeEl.style.display = "block";
      // 如果冲突了，可以自动把主开关禁用了，防止误操作
      document.getElementById("toggle").disabled = true;
    } else {
      noticeEl.style.display = "none";
      document.getElementById("toggle").disabled = false;
    }
  });
}

function el(tag, attrs = {}, children = []) {
  const e = document.createElement(tag);
  Object.entries(attrs).forEach(([k, v]) => {
    if (k === "class") e.className = v;
    else if (k.startsWith("on") && typeof v === "function") e.addEventListener(k.slice(2), v);
    else e.setAttribute(k, v);
  });
  children.forEach(c => e.appendChild(typeof c === "string" ? document.createTextNode(c) : c));
  return e;
}

function showAuth(msg = "") {
  const authBar = document.getElementById("authBar");
  const mainView = document.getElementById("mainView");
  const userArea = document.getElementById("userArea");
  const drawerAuthed = document.getElementById("drawerAuthed");

  // 未登录：显示 authBar，隐藏 mainView / 用户区 / 登录后菜单
  if (authBar) authBar.style.display = "block";
  if (mainView) mainView.style.display = "none";
  if (userArea) userArea.style.display = "none";
  if (drawerAuthed) drawerAuthed.style.display = "none";

  // 清理用户名显示（可选，但建议）
  const usernameText = document.getElementById("usernameText");
  if (usernameText) usernameText.textContent = "";

  // 关闭左侧抽屉（如果存在）
  const drawer = document.getElementById("drawer");
  const backdrop = document.getElementById("drawerBackdrop");
  drawer?.classList.remove("open");
  if (backdrop) backdrop.style.display = "none";
  drawer?.setAttribute("aria-hidden", "true");

  // 写状态
  const s = document.getElementById("authStatus");
  if (s) s.textContent = msg || "";
}

async function showMain(msg = "") {
  const authBar = document.getElementById("authBar");
  const mainView = document.getElementById("mainView");
  const userArea = document.getElementById("userArea");
  const drawerAuthed = document.getElementById("drawerAuthed");

  if (authBar) authBar.style.display = "none";
  if (mainView) mainView.style.display = "block";
  if (userArea) userArea.style.display = "flex";
  if (drawerAuthed) drawerAuthed.style.display = "block";

  // 👇 从 storage 读取用户名
  try {
    const { username } = await chrome.storage.local.get("username");
    if (username) {
      const usernameText = document.getElementById("usernameText");
      if (usernameText) usernameText.textContent = username;
    }
  } catch (e) {
    console.warn("read username failed", e);
  }

  // 关闭左侧抽屉
  const drawer = document.getElementById("drawer");
  const backdrop = document.getElementById("drawerBackdrop");
  drawer?.classList.remove("open");
  if (backdrop) backdrop.style.display = "none";
  drawer?.setAttribute("aria-hidden", "true");

  const status = document.getElementById("status");
  if (status && msg) status.textContent = msg;
}



function render(state) {
  const list = document.getElementById("list");
  const status = document.getElementById("status");
  const toggle = document.getElementById("toggle");

  if (!list || !status || !toggle) return;
  //vip
  const badge = document.getElementById("vipBadge");
  if (badge) {
    if (state.loggedIn && state.token) {
      const v = Number(state.vip || 0);

      badge.style.display = "inline-block";
      badge.textContent = v > 0 ? `VIP${v}` : "VIP0";

      // 清理旧状态
      badge.classList.remove("vip-0", "vip-gold");

      // 根据等级上色
      if (v > 0) {
        badge.classList.add("vip-gold");
      } else {
        badge.classList.add("vip-0");
      }
    } else {
      badge.style.display = "none";
    }
  }


  toggle.checked = !!state.enabled;
  list.innerHTML = "";

  const uv = Number(state.vip || 0);

  // 应用同样的阶梯过滤规则
  const filtered = state.proxies.filter(p => {
    const nv = Number(p.vip || 0);
    if (uv === 0) return nv === 0;
    return nv > 0 && uv >= nv;
  });

  for (const p of filtered) {
    const h = state.health?.[p.id];
    const alive = h?.alive === true;
    const dead = h?.alive === false;

    const dot = el("div", { class: `dot ${alive ? "alive" : dead ? "dead" : ""}` });

    const main = el("div", {}, [
      el("div", {}, [`${p.name || p.id}`]),
      el("div", { class: "meta" }, [
        `${p.scheme}://${p.host}:${p.port}  ${h ? `• ${alive ? "OK" : "DOWN"} • ${h.rtt}ms` : "• 未检测"}`
      ])
    ]);

    const item = el("div", {
      class: `item ${state.selectedId === p.id ? "selected" : ""}`,
      onclick: async () => {
        await chrome.runtime.sendMessage({ type: "SELECT", selectedId: p.id });
        const st = await loadState();
        render(st);
      }
    }, [dot, main]);

    list.appendChild(item);
  }

  status.textContent = `已加载 ${state.proxies.length} 个代理`;
}

async function refreshUI() {
  const st = await loadState();
  render(st);
}


async function doLogout() {
  // 1. 增加 vip 字段的清理，确保不会影响下个用户的 VIP 判断
  await chrome.storage.local.remove(["loggedIn", "email", "username", "token", "vip", "health", "failStats"]);
  
  // 2. 强制将状态设为关闭，防止新用户登录前代理逻辑在后台盲目运行
  await chrome.storage.local.set({ enabled: false });

  // 3. 发送消息通知后台。注意：我们希望后台在收到消息时执行 chrome.proxy.settings.clear()
  try {
    await chrome.runtime.sendMessage({ type: "AUTH_UPDATED" });
  } catch (e) {
    console.warn("通知后台失败，可能 Service Worker 已休眠:", e);
  }

  showAuth("未登录");

  // 4. 打开登录页
  const url = chrome.runtime.getURL("pages/user/login.html");
  await chrome.tabs.create({ url });
}

document.addEventListener("DOMContentLoaded", async () => {
  checkProxyConflict();
  const st = await loadState();

  // 未登录：只显示 登录/注册 两按钮
  if (!st.loggedIn) {
    showAuth("请先登录,或注册后免费使用基础功能");

    const openLogin = document.getElementById("openLogin");
    const openRegister = document.getElementById("openRegister");

    if (openLogin) {
      openLogin.addEventListener("click", async () => {
        const url = chrome.runtime.getURL("pages/user/login.html");
        await chrome.tabs.create({ url });
      });
    }

    if (openRegister) {
      openRegister.addEventListener("click", async () => {
        const url = chrome.runtime.getURL("pages/user/register.html");
        await chrome.tabs.create({ url });
      });
    }

    return;
  }

  // 已登录：显示主界面 + 绑定你原来的事件
  showMain(`已登录：${st.username || ""}`);
  await refreshUI();

  document.getElementById("toggle")?.addEventListener("change", async (e) => {
    await chrome.runtime.sendMessage({ type: "TOGGLE", enabled: e.target.checked });
    await refreshUI();
  });

  document.getElementById("openWelcome").onclick = () => {
    chrome.tabs.create({ url: chrome.runtime.getURL("pages/welcome.html") });
  };
  document.getElementById("openPay").onclick = () => {
    chrome.tabs.create({ url: chrome.runtime.getURL("pages/upgrade.html") });
  };  
  document.getElementById("logout")?.addEventListener("click", doLogout);

  document.getElementById("openWhitelist").onclick = () => {
    chrome.tabs.create({ url: chrome.runtime.getURL("pages/whitelist.html") });
  };
  document.getElementById("openHealth")?.addEventListener("click", async () => {
    const url = chrome.runtime.getURL("pages/health.html");
    await chrome.tabs.create({ url });
  });

document.getElementById("refresh")?.addEventListener("click", async (e) => {
  const btn = e.currentTarget;
  
  // 1. 防止重复点击
  if (btn.classList.contains("spin")) return;

  btn.classList.add("spin");

  try {
    // 2. 发送消息并等待后台确认“已启动任务”
    // 注意：后台的 REFRESH 分支不应该 await checkAllHealth，而是直接返回 ok
    const response = await chrome.runtime.sendMessage({ type: "REFRESH" });
    
    if (response?.ok) {
      // 3. 立即刷新一次 UI，显示从 storage 拿到的新列表（此时可能还没测速完）
      if (typeof render === "function") {
        const state = await loadState();
        render(state);
      }
    }
  } catch (err) {
    console.error("刷新请求失败:", err);
  } finally {
    // 4. 增加一个最短旋转时间（如 1.5 秒），让动画看起来更自然
    // 并且无论成功失败，最后都要移除 spin 类
    setTimeout(() => {
      btn.classList.remove("spin");
    }, 1500);
  }
});
const userBtn = document.getElementById("userButton");
const userMenu = document.getElementById("userMenu");

function closeUserMenu() {
  userMenu?.classList.remove("open");
  userBtn?.setAttribute("aria-expanded", "false");
  userMenu?.setAttribute("aria-hidden", "true");
}

userBtn?.addEventListener("click", (e) => {
  e.stopPropagation();
  const isOpen = userMenu.classList.toggle("open");
  userBtn.setAttribute("aria-expanded", String(isOpen));
  userMenu.setAttribute("aria-hidden", String(!isOpen));
});

document.addEventListener("click", () => closeUserMenu());
userMenu?.addEventListener("click", () => closeUserMenu());


const drawer = document.getElementById("drawer");
const backdrop = document.getElementById("drawerBackdrop");

function openDrawer() {
  drawer?.classList.add("open");
  if (backdrop) backdrop.style.display = "block";
  drawer?.setAttribute("aria-hidden", "false");
}

function closeDrawer() {
  drawer?.classList.remove("open");
  if (backdrop) backdrop.style.display = "none";
  drawer?.setAttribute("aria-hidden", "true");
}

document.getElementById("openMenu")?.addEventListener("click", openDrawer);
document.getElementById("closeMenu")?.addEventListener("click", closeDrawer);
backdrop?.addEventListener("click", closeDrawer);
document.addEventListener("keydown", (e) => {
  if (e.key === "Escape") closeDrawer();
});



});
